// Copyright 1999, 2000 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with ABSOLUTELY NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License
// Version 2.

package com.perdues;

import java.io.*;
import java.net.*;
import java.lang.reflect.Array;


/**
  Static utility methods involving Strings.
  */
public class PString {

  private PString() {}

  //// Public Methods ////

  /**
    Same as object.toString() except for arrays,
    which come out as "{a[1],a[2],...}".  The object may be
    null, which converts to "null".
    */
  public static String toString(Object o) {
    if (o==null) return "null";
    if (o.getClass().isArray()) {
      StringBuffer buf = new StringBuffer();
      buf.append('{');
      int len = Array.getLength(o);
      for (int i=0; i<len; i++) {
        if (i>0) buf.append(',');
        buf.append(Array.get(o, i));
      }
      buf.append('}');
      return buf.toString();
    } else {
      return o.toString();
    }
  }


  /**
    Returns the decimal integer value represented by the String,
    or the given default if the string is null or its
    format is bad.
    */
  public static int intValue(String s, int dfault) {
    if (s==null)
      return dfault;
    try {
      return Integer.parseInt(s);
    } catch(NumberFormatException ex) {
      return dfault;
    }
  }


  /**
     Returns a prefix of subject, up to the first occurrence of
     the delimiter string, or the entire subject if the delimiter
     is not found.
  */
  public static String upTo(String subject, String delimiter) {
    int pos = subject.indexOf(delimiter);
    if (pos<0) {
      return subject;
    } else {
      return subject.substring(0, pos);
    }
  }


  /**
     Returns a prefix of subject, up to the last occurrence of
     the delimiter string, or the entire subject if the delimiter
     is not found.
  */
  public static String upToLast(String subject, String delimiter) {
    int pos = subject.lastIndexOf(delimiter);
    if (pos<0) {
      return subject;
    } else {
      return subject.substring(0, pos);
    }
  }


  /**
     Returns a suffix of the subject, starting just after the
     first occurrence of the delimiter string.  If the delimiter
     is not found, returns the entire subject without copying it.
  */
  public static String after(String subject, String delimiter) {
    int pos = subject.indexOf(delimiter);
    if (pos<0) {
      return subject;
    } else {
      return subject.substring(pos+delimiter.length());
    }
  }


  /**
     Returns a suffix of the subject, starting just after the
     last occurrence of the delimiter string.  If the delimiter
     is not found, returns the entire subject without copying it.
  */
  public static String afterLast(String subject, String delimiter) {
    int pos = subject.lastIndexOf(delimiter);
    if (pos<0) {
      return subject;
    } else {
      return subject.substring(pos+delimiter.length());
    }
  }


  /**
     Pad a string on the right to the specified width in characters,
     using the specified pad character and
     returning the string with added padding.  Never truncate.
  */
  public static String padRight(String s, int width, char c) {
    if (s.length()>=width)
      return s;
    StringBuffer buf = new StringBuffer(width);
    buf.append(s);
    for (int l=s.length(); l<width; l++) {
      buf.append(c);
    }
    return buf.toString();
  }


  /**
     Pad a string on the left to the specified width in characters,
     using the specified pad character and
     returning the string with added padding.  Never truncate.
  */
  public static String padLeft(String s, int width, char c) {
    if (s.length()>=width)
      return s;
    StringBuffer buf = new StringBuffer(width);
    for (int l=s.length(); l<width; l++) {
      buf.append(c);
    }
    buf.append(s);
    return buf.toString();
  }


  /**
    Substitute a new value in place of all occurrences one substring
    within a subject string.
    */
  public static String subst(String subject, String old, String nu) {
    int pos = 0;
    int oldlen = old.length();
    StringBuffer buf = new StringBuffer();
    while (true) {
      int newpos = subject.indexOf(old, pos);
      if (newpos<0) break;
      buf.append(subject.substring(pos, newpos));
      buf.append(nu);
      pos = newpos+oldlen;
    }
    buf.append(subject.substring(pos));
    return buf.toString();
  }


  /**
     Read all remaining input from the InputStream through
     the default character encoding, and return it as a String.
  */
  public static String readAll(InputStream ins) throws IOException {
    return readAll(new InputStreamReader(ins));
  }


  /**
     Reads all remaining input from the Reader and returns
     it as a String.
  */
  public static String readAll(Reader reader) throws IOException {
    char[] chars = new char[4*1024];
    StringBuffer buf = new StringBuffer();
    while (true) {
      int n = reader.read(chars, 0, chars.length);
      if (n<0)
	break;
      buf.append(chars, 0, n);
    }
    return buf.toString();
  }


  /**
     Reads and returns the contents of a URL as a String.
     The content should be of some text type, e.g. text/plain
     or text/html.  This uses Java's default encoding to convert
     bytes to characters.
   */
  public static String readAll(URL url) throws IOException {
    InputStream in = null;
    URLConnection conn0 = url.openConnection();
    if (!(conn0 instanceof HttpURLConnection)) {
      // This condition should never happen because of the
      // URL check during initialization.
      throw new IllegalArgumentException
	("Connection to "+url+" must be via HTTP, not "
	 +conn0.getClass().getName());
    }
    HttpURLConnection conn = (HttpURLConnection)conn0;
    conn.setAllowUserInteraction(false);
    conn.setUseCaches(false);
    conn.connect();
    int code = conn.getResponseCode();
    if (code<200 || code>=300) {
      throw new IllegalStateException("Response code was "+code);
    }
    try {
      in = conn.getInputStream();
      String s = readAll(in);
      return s;
    } finally {
      if (in!=null)
	in.close();
    }
  }

  /**
    For testing.
    */
  public static void main(String[] args) {
    try {
      String[] ss = {"Mary had a little lamb."," ","&nbsp;"};
      for (int i=0; i<ss.length; i+=3) {
	System.out.println("Subst Target: '"+ss[i]+"' old: '"+ss[i+1]
			   +"' new: '"+ss[i+2]+"'");
	System.out.println("Result is: '"+subst(ss[i],ss[i+1],ss[i+2])+"'");
      }
      if (args.length>0) {
	URL url = new URL(args[0]);
	System.out.println("URL="+url+", contents is:");
	System.out.println("Host="+url.getHost());
	System.out.println("InetAddress="+InetAddress.getByName(url.getHost()));
	System.out.println(PString.readAll(url));
      }
    } catch(Throwable ex) {
      ex.printStackTrace(System.err);
    }
  }
}
