// Copyright 2000 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with ABSOLUTELY NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License
// Version 2.

package com.perdues;

import java.util.*;
import java.io.*;


/**
  Recompile .java files that need it anywhere under the current
  directory, using javac.  The condition is the same used by "make":
  the corresponding .class file does not exist or is older.
  */
public class Recompiler {

  /**
     Takes no command line arguments.
   */
  public static void main(String[] args) {
    // Vector of strings naming the .java files to be
    // recompiled.
    Vector toRecompile = new Vector();
    // Call the work method on the current directory.
    update(new File("."), toRecompile);
    // Success if nothing needs to be recompiled.
    if (toRecompile.size()==0) {
      System.exit(0);
    }
    // Get ready to turn the Vector into a command for "exec".
    toRecompile.insertElementAt("javac", 0);
    String[] command = new String[toRecompile.size()];
    toRecompile.copyInto(command);
    // Display the command.
    for (int i=0; i<command.length; i++) {
      System.out.print(command[i]);
      System.out.print(" ");
    }
    System.out.println();
    // Run the compiler, passing its standard output and standard error.
    // Wait for completion of the process and all its output.
    // Return the same exit status as the compiler.
    Runtime run = Runtime.getRuntime();
    try {
      Process javac = run.exec(command);
      BytePump out = new BytePump(javac.getInputStream(), System.out);
      BytePump err = new BytePump(javac.getErrorStream(), System.err);
      out.start();
      err.start();
      out.join();
      err.join();
      javac.waitFor();
      System.exit(javac.exitValue());
    } catch(Throwable ex) {
      System.err.println(ex);
      System.exit(1);
    }
  }

  /**
     Recursive main work method.  This checks for .java files that
     need recompilation and recurs into all subdirectories.
   */
  public static void update(File dir, Vector toRecompile) {
    String separator = File.separator;
    String[] files = dir.list();
    for (int i=0; i<files.length; i++) {
      String filename = files[i];
      File file = new File(dir, filename);
      if (filename.endsWith(".java")) {
	String basename = filename.substring(0, filename.length()-5);
	File classFile = new File(dir, basename+".class");
	if (!(classFile.exists()
	      && classFile.lastModified()>=file.lastModified())) {
	  toRecompile.addElement(file.getPath());
	}
      } else if (file.isDirectory()) {
	update(file, toRecompile);
      }
    }
  }
  
  /**
    Copies from InputStream to OutputStream
    until EOF.  IOExceptions result in a RuntimeException
    thrown from "run".
    */
  public static class BytePump extends Thread {

    public BytePump(InputStream in, OutputStream out) {
      this.in = in;
      this.out = out;
    }

    /**
      Implementation of Thread.run.
      */
    public void run() {
      byte[] buf=new byte[1000];
      try {
	while (true) {
	  int n=in.read(buf,0,buf.length);
	  if (n<0) break;
	  out.write(buf,0,n);
	}
      } catch (IOException ex) {
	throw new RuntimeException(ex.toString());
      }
    }

    private InputStream in;
    private OutputStream out;
  }

}

