// Copyright 1999, 2000 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with ABSOLUTELY NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License
// Version 2.
//

package com.perdues;

import java.util.Vector;


/**
 * Class to support periodic jobs passed to it as
 * Runnables.  This will run each one at fixed intervals
 * given in seconds.  Spawns a new Thread for each job
 * it runs.  This class's interface is based on
 * com.bitmechanic.util.Scheduler, written by James Cooper.
 * <P>
 * If you would like to run a task once per minute,
 * implemented by an instance of MyRunnable, you need only do:
 * <PRE>
 *   Scheduler.defaultScheduler().scheduleJob(new MyRunnable(), 60);
 * </PRE>
 * It will run in a new Thread every 60 seconds.
*/
public class Scheduler extends Thread {

  /**
     Turn this flag on to receive debugging information
     on System.err.
   */
  public static boolean debug = false;

  /**
     Creates but does not start a new Scheduler.
  */
  public Scheduler() {
    setName("Job Scheduler");
  }


  /**
     Returns the default Scheduler, creating and starting it if this
     has not already been done.  The default Scheduler runs as
     a daemon thread.  If all "user" (non-daemon) threads exit,
     the Java VM will exit also.
  */
  public static synchronized Scheduler defaultScheduler() {
    if (sched!=null)
      return sched;
    sched = new Scheduler();
    sched.setDaemon(true);
    sched.setName("Default Scheduler");
    sched.start();
    if (debug)
      System.err.println("Started defaultScheduler");
    return sched;
  }

  // Holds the default Scheduler.
  private static Scheduler sched = null;


  /**
     Adds a Runnable to this Scheduler, to be run every sleepInterval seconds.
   */
  public synchronized void scheduleJob(Runnable job, int sleepInterval) {
    if (debug)
      System.err.println("Adding job, runs every "+sleepInterval+" seconds.");
    ScheduledJob j = new ScheduledJob(job, sleepInterval);
    insert(j);
  }

  /**
     Removes a periodic job from the list.
   */
  public synchronized void removeJob(Runnable job) {
    for(int i = 0; i < jobs.size(); i++) {
      ScheduledJob j = (ScheduledJob)jobs.elementAt(i);
      if(j.runnable == job) {
	jobs.removeElementAt(i);
	// No need to wake up the scheduler just to tell it that
	// it might not have to wake up so soon...
	return;
      }
    }
  }

  /**
     Implementation of Runnable; not intended to be called
     directly from user code.  This is the main loop;
     waits for jobs to be ready to run and runs each in its
     own thread.
   */
  public synchronized void run() {
    if (debug)
      System.err.println(this+" starting.");
    try {
      while (true) {
	long now = System.currentTimeMillis();
	if (jobs.size()==0) {
	  if(debug) System.err.println("Waiting for a task");
	  // No candidate jobs; wait.
	  this.wait();
	} else {
	  long waitTime = ((ScheduledJob)jobs.elementAt(0))
	    .wakeuptime-now;
	  if (waitTime>0) {
	    if(debug) System.err.println("Sleeping for " + waitTime + "ms");
	    // Wait until a job might be ready to run.
	    this.wait(waitTime);
	  } else {
	    // Found a job ready to run; start it.
	    ScheduledJob job = (ScheduledJob)jobs.elementAt(0);
	    Thread t = new Thread(job.runnable);
	    t.setDaemon(true);
	    t.setName("Job "+ ++jobID);
	    if(debug) System.err.println("Running job");
	    t.start();
	    jobs.removeElement(job);
	    job.reset(now);
	    insert(job);
	  }
	}
      }
    } catch(InterruptedException e) {
      if (debug) {
        System.err.println("Interrupted Scheduler, quitting.");
        e.printStackTrace(System.err);
      }
      // If interrupted during a wait, JavaSoft concept is to terminate it.
      return;
    } catch(Throwable e) {
      if (debug) {
        // This code should be redundant with default ThreadGroup
        // uncaughtException handling.
        System.err.println("Error in Scheduler, quitting.");
        e.printStackTrace(System.err);
        return;
      }
    }
  }


  /**
     Returns the number of periodic jobs currently managed
     by this Scheduler.
  */
  public int getJobCount() {
    return jobs.size();
  }


  //// Private Methods ////

  /**
     Insert a job into the queue.  If it is inserted at the front,
     wake up the Scheduler thread for shorter sleep or immediate
     action.
  */
  private void insert(ScheduledJob job) {
    long next = job.wakeuptime;
    int njobs = jobs.size();
    for(int i=0; i<=njobs; i++) {
      if (i>=njobs || next<((ScheduledJob)jobs.elementAt(i)).wakeuptime) {
	jobs.insertElementAt(job, i);
	if (i==0) {
	  notify();
	}
	return;
      }
    }
    throw new IllegalStateException("Internal error in Scheduler.");
  }


  //// Private Data ////


  private Vector jobs = new Vector();

  // Last used ID for a job.
  private int jobID = 0;
  

  //// Private Class ////

  static class ScheduledJob {

    Runnable runnable;
    long wakeuptime;
    int  sleep_interval;

    public ScheduledJob(Runnable job, int sleep_interval) {
      this.runnable = job;
      this.sleep_interval = sleep_interval;
      reset(System.currentTimeMillis());
    }

    public void reset(long now) {
      wakeuptime = now + (sleep_interval * 1000);
    }

  }

}

