// Copyright 1999, 2000 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License
// Version 2.

package com.perdues;

import java.util.*;
import java.text.SimpleDateFormat;


/**
  The StyleDate class makes it convenient to combine
  Calendar and time arithmetic with formatting.
  Combines the functions of java.util.Date, java.sql.Date,
  SimpleDateFormat, and Calendar with a single interface.
  <p>
  A StyleDate maintains a current date/time, and provides
  methods to set the various date/time fields and to
  add (or subtract) from its current value.  The toString
  method formats the current date/time according to the
  current format.
  <p>
  You can set the day within the current week, the day of
  the month, the month, or the year directly, and set the
  time to the beginning of the day.  You can add positive
  or negative numbers of days, months, years, or milliseconds
  to the internal date/time with calendar-based results.
  <p>
  All the calendar fields are available with normalized data
  at all times.  For example, if you set the date/time to the
  beginning of the month, then add a month, the date/time
  is the beginning of the next month.
  <p>
  The TimeZone and Locale are configurable as well as the format.
  */
public class StyleDate {

  //// Constructors ////

  /**
    Creates a StyleDate with the current date and time.
    */
  public StyleDate() {}


  /**
    Creates a StyleDate for the given Date.
    */
  public StyleDate(Date d) {
    cal.setTime(d);
  }


  /**
    Creates a StyleDate for the given absolute
    time in milliseconds.
    */
  public StyleDate(long millis) {
    cal.setTime(new Date(millis));
  }


  /**
    Creates a StyleDate with the current
    date/time and the given format, which it
    localizes.
    */
  public StyleDate(String fmt) {
    format.applyLocalizedPattern(fmt);
  }

  /**
    Creates a StyleDate with the same date/time,
    format, locale, and timezone as the one given.
    */
  public StyleDate(StyleDate s) {
    cal = (Calendar)s.cal.clone();
    format = (SimpleDateFormat)s.format.clone();
  }


  /**
    Create a StyleDate with a particular format
    and Date.
    */
  public StyleDate(Date d, String fmt) {
    cal.setTime(d);
    format.applyLocalizedPattern(fmt);
  }


  //// Assignment ////

  /**
    Sets the date/time to the given Date.
    */
  public void setDate(Date d) {
    cal.setTime(d);
  }


  /**
    Sets the date/time to the formatted date, parsing
    leniently according to the current format.
    */
  public void setDate(String date) throws java.text.ParseException {
    cal.setTime(format.parse(date));
  }


  /**
    Sets the date/time to an absolute value in milliseconds.
    */
  public void setMillis(long m) {
    cal.setTime(new Date(m));
  }


  //// Configuration ////

  /**
    Sets the format (does not localize it, whatever that implies).
    Just as if calling the SimpleDateFormat constructor with a format
    parameter.
    */
  public void setFormat(String fmt) {
    format.applyPattern(fmt);
  }


  /**
    Sets the format, translating it to a localized version.
    This appears to translate some of the pattern characters
    from standard ones for the default format into the corresponding
    characters for the Formatter's locale.
    */
  public void setLocalizedFormat(String fmt) {
    format.applyLocalizedPattern(fmt);
  }


  public void setTimeZone(TimeZone z) {
    cal.setTimeZone(z);
    format.setTimeZone(z);
  }


  public void setLocale(Locale loc) {
    format = new SimpleDateFormat(format.toPattern(), loc);
    Date d = asDate();
    cal = Calendar.getInstance(loc);
    setDate(d);
  }


  //// Conversion to Date and milliseconds ////

  /**
    Return the current date/time as a Date.
    */
  public Date asDate() {
    return cal.getTime();
  }


  /**
    Returns the current date/time in milliseconds.
    */
  public long asMillis() {
    return cal.getTime().getTime();
  }


  //// Formatting ////

  /**
    Format the currently-set date/time for SQL.
    */
  public String sqlFormat() {
    return new java.sql.Date(asMillis()).toString();
  }


  /**
    Format the currently-set date/time.
    */
  public String format() {
    return format.format(asDate());
  }


  /**
    Formats the currently-set date/time.
    */
  public String toString() {
    return format();
  }


  //// Partial Assignment ////

  private static final int DAY = Calendar.DAY_OF_MONTH;

  /**
    Sets the day of the month.  Modifies the effective
    time of the of the calendar to match.
    */
  public void setDay(int d) {
    // Setting the day of month should override other information
    // about the day.
    cal.set(DAY, d);
  }


  /**
    Returns the first day of the week for the locale.
    */
  public int getFirstDayOfWeek() {
    return cal.getFirstDayOfWeek();
  }


  private static final int WKDAY = Calendar.DAY_OF_WEEK;

  /**
    Sets the day of the week to the given constant within the
    current week, where the current week is determined by
    the locale's notion of "first day" of a week.
    For example, setDayOfWeek(Calendar.SUNDAY) sets the date/time
    to Sunday of "this" week, today if the week begins on Sunday,
    otherwise last Sunday.
    */
  public void setDayOfWeek(int d) {
    int current = cal.get(WKDAY);
    int start = cal.getFirstDayOfWeek();
    if (start>current) current -= 7;
    // Move back to a Sunday and ahead to the desired day.
    cal.add(WKDAY, -current+d);
  }


  /**
    Sets the date/time to the beginning of the day, local time.
    */
  public void setToStartOfDay() {
    cal.set(Calendar.HOUR_OF_DAY, 0);
    cal.set(Calendar.MINUTE, 0);
    cal.set(Calendar.SECOND, 0);
    cal.set(Calendar.MILLISECOND, 0);
  }


  /**
    Sets the month, zero-based counting.
    */
  public void setMonth(int m) {
    cal.set(Calendar.MONTH, m+cal.getMinimum(Calendar.MONTH));
  }


  /**
    Sets the year.
    */
  public void setYear(int y) {
    cal.set(Calendar.YEAR, y);
  }


  //// Other Operations ////

  /**
    Ajust the day forward the given number of days, negative to go back.
    */
  public void addDays(int n) {
    cal.add(Calendar.DAY_OF_MONTH, n);
  }


  /**
    Ajust the month forward the given number of months, negative to go back.
    */
  public void addMonths(int n) {
    cal.add(Calendar.MONTH, n);
  }


  /**
    Ajust the year forward the given number of years, negative to go back.
    */
  public void addYears(int n) {
    cal.add(Calendar.YEAR, n);
  }


  public void addMillis(long m) {
    setMillis(asMillis()+m);
  }

  /**
    Gets an arbitrary Calendar field.  Useful field constants are:
    <UL>
    <LI>Calendar.YEAR
    <LI>Calendar.MONTH (first month is numbered 0)
    <LI>Calendar.DAY_OF_MONTH
    <LI>Calendar.DAY_OF_WEEK (numbered from Sunday=1)
    <LI>Calendar.HOUR_OF_DAY
    <LI>Calendar.HOUR (for 12-hour presentation)
    <LI>Calendar.MINUTE
    <LI>Calendar.SECOND
    <LI>Calendar.MILLISECOND
    </UL>
    */
  public int get(int field) {
    return cal.get(field);
  }


  /**
    Sets an arbitrary Calendar field.  
    */
  public void set(int field, int value) {
    cal.set(field, value);
  }


  /**
    Is this date after the argument?
    */
  public boolean after(Date d) {
    return asMillis()>d.getTime();
  }


  /**
    Is this date after the argument?
    */
  public boolean after(StyleDate d) {
    return asMillis()>d.asMillis();
  }


  /**
    Is this date before the argument?
    */
  public boolean before(Date d) {
    return asMillis()<d.getTime();
  }


  /**
    Is this date before the argument?
    */
  public boolean before(StyleDate d) {
    return asMillis()<d.asMillis();
  }


  //// Private data ////

  // Starts with current date and time.
  private Calendar cal = Calendar.getInstance();

  // Default is lenient, but let's make it explicit:
  { cal.setLenient(true); }


  private SimpleDateFormat format = new SimpleDateFormat("MM/dd/yyyy");

  { format.setLenient(true); }



  //// Testing ////

  public static void main(String[] args) {
    // TimeZone.setDefault(TimeZone.getTimeZone("PST"));
    Date now = new Date();
    StyleDate date = new StyleDate();
    
    java.io.PrintStream out = System.out;

    String[] formats = new String[] {
      "EEEE MMMM d, yyyy"
	};
    for (int i=-1; i<formats.length; i++) {
      date.setDate(now);
      if (i>=0) date.setFormat(formats[i]);
      out.println("Now: "+date.format());
      date.setDayOfWeek(date.getFirstDayOfWeek());
      out.println("Beginning of week: "+date.format());
      date.setDate(now);
      date.setDay(1);
      out.println("Beginning of month: "+date);
      date.addMonths(-1);
      out.println("Beginning of last month: "+date);
      date.setMonth(0);
      out.println("Beginning of January: "+date);
      date.addYears(-1);
      out.println("Beginning of last year: "+date);
      date.addDays(-14);
      out.println("Two weeks before: "+date);
      date.addDays(-14);
      out.println("Two weeks before: "+date);
    }
  }

}
