// Copyright 2000 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with ABSOLUTELY NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License
// Version 2.

package com.perdues;

import java.io.*;

/**
  Utility class for running a subprocess synchronously from Java
  and retrieving the contents of System.out and/or system.err.

  @author Cris Perdue <cris@perdues.com>
  */
public class SyncExec {

  /**
    This class is not instantiable.
    */
  private SyncExec() {}


  /**
     Runs a process using Runtime.exec().  Collects all output
     on System.out into a String and returns it, waiting for
     all output to System.out and System.err, and for completion
     of the child process.  If the exit status is nonzero or
     there is any output to System.err, this throws a
     com.perdues.SyncExec.Exception containing the exit status
     and error output.
  */
  public static String exec(final String[] cmd)
  throws IOException, InterruptedException {
      StringWriter w = new StringWriter();
      StringWriter e = new StringWriter();
      int status = exec(cmd, w, e);
      if (status!=0 || e.getBuffer().length()>0)
	throw new Exception(status, e.toString());
      else
	return w.toString();
  }


  /**
     Runs a process using Runtime.exec().  Collects all output
     on System.out and System.err, passing it to the two Writer
     parameters.  Waits for all output and process completion,
     writing to the two Writers via the system default character encoding,
     then returns the process exit status.
     <P>
     Note the if one of the Writers blocks, this may block the child
     process.
  */
  public static int exec(final String[] cmd, Writer output, Writer errors)
    throws IOException, InterruptedException {
      Process program = Runtime.getRuntime().exec(cmd);
      CharPump outpump
        = new CharPump(new InputStreamReader(program.getInputStream()), output);
      CharPump errpump
        = new CharPump(new InputStreamReader(program.getErrorStream()), errors);
      outpump.start();
      errpump.start();
      outpump.join();
      errpump.join();
      return program.waitFor();
  }


  // Inner class, copies from Reader to Writer.
  public static class CharPump extends Thread {

    public CharPump(Reader in, Writer out) {
      this.in = in;
      this.out = out;
    }

    public void run() {
      char[] buf=new char[1024];
      try {
        while (true) {
          int n=in.read(buf,0,buf.length);
          if (n<0) break;
          out.write(buf,0,n);
        }
	out.flush();
      } catch (IOException ex) {
        throw new RuntimeException("IOException");
      }
    }

    private Reader in;
    private Writer out;
  }


  // Inner class SyncExec.Exception for error reporting.
  public static class Exception extends RuntimeException {

    public Exception(int status, String output) {
      super(output);
      this.status = status;
    }

    public int getStatus() {
      return status;
    }

    public String toString() {
      return this.getClass().getName()+": exit "+status+": "+getMessage();
    }

    private int status;

  }


  /**
    Main routine, for testing.
    The name of the executable to run and its
    arguments are the command line arguments.
    */
  public static void main(String[] args) {
    try {
      System.out.println(exec(args));
    } catch(java.lang.Exception ex) {
      System.err.println(ex.toString());
    }
  }

}
