//
// Copyright 1999 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with ABSOLUTELY NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License
// Version 2.
//

package com.perdues;

import java.util.*;
import java.io.*;


/**
  Abstract class that provides a framework for programs
  that filter textual input to an output.  Subclass it
  and define handleLine(String) and a main program to make
  a complete command-line application.
  */
public abstract class TextFilter {

  /**
    Convenience method that calls the filter method with File
    arguments.  If there are no arguments or the first argument
    is "-", the input is System.in.  If there are not two
    arguments, the output is System.out.  Otherwise input and
    output are the files named by the first two elements of the
    args array.
   */
  public void filter(String[] args) throws IOException {
    int length = args.length;
    File in = null;
    File out = null;
    if (length>=1 && !"-".equals(args[0]))
      in = new File(args[0]);
    if (length>=2)
      out = new File(args[1]);
    filter(in, out);
  }
    

  /**
    This method opens up a BufferedReader for input from the
    given input file and a PrintWriter for output to the given
    output file.  If the input file is null, it uses System.in
    instead.  If the output file is null, this uses System.out
    in its place.
    <P>
    This reads lines from input until end of file.  On each
    line it calls handleLine, passing it the current line.
    On end of file or if it catches an exception it closes its input
    and output files, except it will not close System.in or System.out.
    <P>
    Since reading of lines is done by BufferedReader.readLine,
    there is no detection of an unterminated last input line.
    */
  public void filter(File in, File out) throws IOException {
    BufferedReader r = null;
    try {
      if (in!=null) {
	FileReader fr = new FileReader(in);
	r = new BufferedReader(fr);
      } else {
	r = new BufferedReader(new InputStreamReader(System.in));
      }

      if (out!=null) {
	FileWriter fw = new FileWriter(out);
	w = new PrintWriter(fw);
      } else {
	w = new PrintWriter(System.out);
      }

      String line;
      while ((line = r.readLine())!=null) {
	handleLine(line);
      }
      handleEOF();
    } finally {
      if (in!=null)
	r.close();
      if (out==null)
	w.flush();
      else
	w.close();
    }
  }


  /**
    Filter calls this on each line of input.  This
    method handles the line, for example writing out
    the transformed input.
    */
  public abstract void handleLine(String line);


  /**
    Override this method to define an action to
    do at end of input.  This should not close
    the input or output.
    */
  public void handleEOF() {}
    

  /**
    This variable supports output.  If the output file is non-null,
    it will be open to that file.  If the output file is null, this
    will be open to System.out.
    */
  protected PrintWriter w;

}
