// Copyright 1999, 2000 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with ABSOLUTELY NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License.
// See http://www.gjt.org/doc/lgpl/license.html for details.

package com.perdues.db;

import java.util.*;
import java.sql.*;
import com.perdues.XMLTag;


public class ColumnInfo {

  //// CONSTRUCTORS ////

  /**
     This creates a ColumnInfo from a TableInfo object
     and an XMLTag.  The tag should be of type "field".
     <P>
     XML attributes:
     <UL>
     <LI>name (required)
     <LI>type (required)
     <LI>caption (optional)
     <LI>required (optional)
     </UL>
     Specific field types may allow or require additional
     attributes in the XML tag.  For example, some types of
     fields may have a maximum length.  Some specific field types
     may have elements as well as attributes.
    */
  public ColumnInfo(TableInfo table, XMLTag tag) {
    if (!"field".equals(tag.getType())) {
      throw new IllegalArgumentException
	("Within table tag, expecting 'field' tag instead of "
	 +tag.getType());
    }
    this.table = table;
    this.tag = tag;

    name = tag.requiredProperty("name");
    type = tag.requiredProperty("type");
    caption = tag.getProperty("caption");

    isRequired = false;
    if ("true".equals(tag.getProperty("required")))
      isRequired = true;
  }


  //// PUBLIC METHODS ////

  public String toString() {
    return "<ColumnInfo "+getName()
      +" "+getType()+(isRequired ? " required" : "")+">";
  }


  /**
    Returns just the (unqualified) name of the column.
    */
  public final String getName() {
    return name;
  }


  /**
    Returns the fully-qualified name of this column,
    currently <table>.<name>.  If the table has a shorthand,
    uses the shorthand.
    */
  public final String getFullName() {
    return table.getShorthand()+"."+getName();
  }


  /**
     Returns the user-oriented caption for the column.
  */
  public final String getCaption() {
    return caption;
  }


  /**
    Returns the TableInfo this column belongs to.
    */
  public final TableInfo getTable() {
    return table;
  }


  /**
    Return the Schema for this Column.
    */
  public final DBInfo getDatabase() {
    return table.getDatabase();
  }


  /**
    Return the type string for the column as set from
    the Schema information.
    */
  public final String getType() {
    return type;
  }


  /**
     True if the column has the "required" attribute.
  */
  public boolean isRequired() {
    return isRequired;
  }


  public DBTypeHandler getTypeHandler() {
    DBInfo db = getDatabase();
    Schema s = db.getSchema();
    // String type = getType();
    // System.out.println("type="+type);
    // DBTypeHandler handler = s.getDBHandler(type);
    return getDatabase().getSchema().getDBHandler(getType());
  }


  /**
     Returns the value of the named attribute of
     the XMLTag that defined this column, or null if
     none was specified.
   */
  public String get(String key) {
    return tag.getProperty(key);
  }


  /**
     Returns an array of XMLTag objects for the elements
     of the XML tag that defined this column.
   */
  public XMLTag[] getElements() {
    return tag.getElements();
  }


  /**
    Get an Enum field's choices as an array.  These are the value of
    the "choices" property, treated as a list delimited by
    "|" (vertical bars).
  */
  public final String[] getChoices() {
    if (!"Enum".equals(getType()))
      throw new IllegalArgumentException("Not an Enum field: "+name);
    String choices = tag.requiredProperty("choices");
    StringTokenizer t = new StringTokenizer(choices, "|");
    String[] result = new String[t.countTokens()];
    for (int i=0; i<result.length; i++) {
      result[i] = t.nextToken();
    }
    return result;
  }
 
    
  /**
     Return the choices, with an extra string prepended.
  */
  public final String[] getChoicesPlus(String extra) {
    return prepend(extra, getChoices());
  }


  //// PRIVATE METHODS ////

  /**
     Make a new array with a new element at the front.
  */
  private static String[] prepend(String s, String[] a) {
    String[] result = new String[a.length+1];
    result[0] = s;
    for (int i=0; i<a.length; i++) {
      result[i+1] = a[i];
    }
    return result;
  }


  //// PRIVATE DATA ////

  private TableInfo table;
  private XMLTag tag;

  private String name;
  private String type;
  private boolean isRequired = false;
  private String caption;

}
