// Copyright 1999, 2000 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License.
// See http://www.gjt.org/doc/lgpl/license.html for details.

package com.perdues.db;

import java.util.*;
import java.io.*;
import java.lang.reflect.Array;
import com.perdues.PString;
import com.perdues.XP;
import com.perdues.XMLTag;
import com.perdues.db.typeHandlers.*;


public class DBInfo {

  //// CONSTRUCTORS ////

  /**
     Build a DBInfo for the given schema, defined with
     the information from an XMLTag of type "db".
     <P>
     XML attributes are:
     <LI>name (required)
     <UL>
     Elements should be of type table.
     </UL>
   */
  public DBInfo(Schema schema, XMLTag tag) {
    if (!tag.getType().equals("db")) {
      throw new IllegalArgumentException
	("Can't define a database from a "+tag.getType()+" tag.");
    }
    this.schema = schema;
    this.name = tag.requiredProperty("name");
    XMLTag[] ttags = tag.getElements();
    for (int i=0; i<ttags.length; i++) {
      XMLTag ttag = ttags[i];
      String name = ttag.requiredProperty("name");
      TableInfo table = new TableInfo(this, ttag);
      tables.put(name, table);
      tables.put(table.getShorthand(), table);
    }
  }


  DBInfo(Schema schema, DBInfo info) {
    this.schema = schema;
    name = info.name;
    for (Enumeration en = tables.keys(); en.hasMoreElements(); ) {
      String key  = (String)en.nextElement();
      tables.put(key, info.tables.get(key));
    }
  }


  //// PUBLIC METHODS ////

  /**
     Get the name of the database.
  */
  public String getName() {
    return name;
  }


  /**
     Get the Schema for this DBInfo.
  */
  public Schema getSchema() {
    return schema;
  }


  /**
    Returns the TableInfo with the given name
    or shorthand, or errs if there is none.
    */
  public TableInfo getTable(String name) {
    TableInfo table = (TableInfo)tables.get(name);
    if (table==null)
      throw new IllegalArgumentException("No table named "+name);
    return table;
  }


  /**
    Return a ColumnInfo given a column name
    qualified by a table name or shorthand.
    */
  public ColumnInfo getColumn(String name) {
    int dot = name.indexOf('.');
    if (dot<1) throw new IllegalArgumentException
		 ("Schema: bad column name: "+name);
    TableInfo table = getTable(name.substring(0, dot));
    return table.getColumn(name.substring(dot+1));
  }


  //// PRIVATE DATA ////

  // Name of the database
  private String name;

  // Maps from unqualified table name to TableInfo.
  private Hashtable tables = new Hashtable();

  // The schema for this DBInfo.  Accessible to Schema copier.
  Schema schema;

}
