// Copyright 1999-2001 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with ABSOLUTELY NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License.
// See http://www.gjt.org/doc/lgpl/license.html for details.

package com.perdues.db;

import java.util.*;
import java.sql.*;
import java.text.*;
import javax.servlet.http.*;
import java.util.Date;
import com.perdues.WrappedException;


/**
   This class extends HTForm with methods to read, insert, update,
   and delete records in a SQL database, and to validate data entered
   in an HTML form for insert or update of a database record.  All of
   these methods automatically match Form fields with database fields
   with corresponding names.  Database reads also do simple joins
   automatically based on identified foreign keys.
   <P>
   Validation uses extended database schema information that can include
   application types such as email, date, phone number, and so on.
   <P>
   Some of the methods associate database fields with form fields.
   Dots in qualified database field names match with double-underscores
   in the form field name.   

   @author Crispin Perdue <cris@perdues.com>
 */
public class Form extends com.perdues.web.HTForm {


  //// PUBLIC DATA ////

  public static boolean debug = true;

  //// CONSTRUCTORS ////

  /**
    A form gets its initial state from the parameters in the request.
    */
  public Form(HttpServletRequest request, Schema schema) {
    super(request);
    this.schema = schema;
  }


  /**
     Constructs an empty Form.
  */
  public Form(Schema schema) {
    super();
    this.schema = schema;
  }


  /**
     Makes a new Form that is a copy of an existing one.
  */
  public Form(Form form) {
    super(form);
    this.schema = form.schema;
  }


  ////////////////////////////////////////////////////////////
  //
  // DATABASE OPERATIONS
  //
  ////////////////////////////////////////////////////////////

  /**
     Reads a row of data from the named table into this
     Form via the given JDBC connection.  This operation
     relies on extended schema information.  This Form
     must already contain a binding for the primary key
     of the named table, and that value determines which
     row this reads.  Reads all fields of the table, but
     will not auto-join.
     <P>
     FIXME: Currently doesn't return anything.
  */
  public void read(String table, Connection conn) {
    read(table, conn, null, null);
  }


  /**
     Reads a row of data from the named table into this
     Form via the given JDBC connection.  This operation
     relies on extended schema information.  This Form
     must already contain a binding for the primary key
     of the named table, and that value determines which
     row this reads.  This only fills fields named in the
     "fields" array, but may automatically join the table
     through foreign key fields to access associated values
     from other tables.
  */
  public synchronized void read
    (String tableName, Connection conn, String[] fields, String[] extras) {
    TableInfo table = schema.getTable(tableName);
    // table.checkForPrimaryKeyField();
    
    if (fields==null)
      fields = allFieldNames(table, noStrings);

    if (extras==null)
      extras = noStrings;

    Query reader = new Query(schema);
    // Put in the table name.
    reader.addTable(table);
    // Add the desired field names.
    Enumeration cols = table.columns();
    while (cols.hasMoreElements()) {
      ColumnInfo column = (ColumnInfo)cols.nextElement();
      reader.addColumn(column);
    }

    // The basic record reader is set up; now build a select
    // statement that gets any additional fields also.
    Query sel = new Query(reader);
    if (extras!=null) {
      for (int i=0; i<extras.length; i++) {
	String field = extras[i];
	// Include fully-qualified names of fields in other
	// tables.
	ColumnInfo column = schema.getColumn(field);
	sel.addColumn(column);
	// FIXME: Is this next line really necessary?
	// sel.addTable(column.getTable());
      }
    }
    sel.addCondition(primaryKeyCheck(table));
    sel.addEquiJoins(table);

    // Read the data into the form.
    ResultSet result = null;
    try {
      String sql = sel.render();
      dbgPrint("Query: "+sql);
      result = conn.createStatement().executeQuery(sql);
      if (result.next()) {
	String[] names = plusFieldNames(fields, extras);

	Vector colsv = new Vector();
	for (int i=0; i<fields.length; i++) {
	  colsv.addElement(table.getColumn(fields[i]));
	}
	for (int i=0; i<extras.length; i++) {
	  colsv.addElement(schema.getColumn(extras[i]));
	}
	for (int k=0; k<names.length; k++) {
	  String name = names[k];
	  // "Dots" in the database field name become "__" in the form
	  // field name.
	  int dot = name.indexOf('.');
	  String formname =
	    dot<0 ? name : name.substring(0,dot)+"__"+name.substring(dot+1);
	  ColumnInfo col = (ColumnInfo)colsv.elementAt(k);
	  col.getTypeHandler()
	    .storeToForm(col, result, name, this, name);
	}
      } else System.err.println("No record with primary key="
				+getValue(table.getPrimaryKey()));
      if (result.next()) {
	System.err.println("More than one row with primary key="
			   +getValue(table.getPrimaryKey()));
      }
    } catch(SQLException ex) {
      System.err.println("Internal error:");
      ex.printStackTrace();
    } finally {
      try {
        if (result!=null) result.close();
      } catch(SQLException ex) { ex.printStackTrace(); }
    }
  }


  /**
    Checks all fields of the form according to the type specified
    for the relevant column of this table.  As usual, each form field
    is matched with the column that has the same name.
    Returns a hashtable that maps from the full name of any column
    that is invalid, to an error message which is a sentence without
    a subject.  The error message displayer has the responsibility
    of filling in the subject.
    <P>
    Only checks form fields with
    names matching column names of the table.  In some applications
    you may wish to first set default values into the form
    where appropriate.
    <P>
    Delegates checking of individual columns to the
    ColumnInfo.validate method, passing in the column's name
    as the form's fieldname.  The type's validation method should not
    give an error if the field is "logically absent" from the form.
    */
  public Hashtable validate(String tableName) {
    Hashtable errs = new Hashtable();
    validate(tableName, errs);
    return errs;
  }


  /**
     Validates this Form using extended schema information.  Checks
     fields that are present in the Form and match
     database field names in the list.  Returns
     a Hashtable of validation information.
  */
  public Hashtable validate(String tableName, String[] fields) {
    Hashtable errs = new Hashtable();
    validate(tableName, errs, fields);
    return errs;
  }


  /**
     A form of validate that takes a Hashtable
     as input and adds any error messages to it.  Any newly-added
     error for a column overrides an existing entry.
  */
  public void validate(String tableName, Hashtable errs) {
    TableInfo table = schema.getTable(tableName);
    Enumeration en = table.columns();
    while (en.hasMoreElements()) {
      ColumnInfo col = (ColumnInfo)en.nextElement();
      col.getTypeHandler().validate(col, this, errs);
    }
  }


  /**
     A form of validate that takes a Hashtable
     as input and adds any error messages to it.  Any newly-added
     error for a column overrides an existing entry.
  */
  public void validate(String tableName, Hashtable errs, String[] fields) {
    TableInfo table = schema.getTable(tableName);
    for (int i=0; i<fields.length; i++) {
      ColumnInfo col = table.getColumn(fields[i]);
      col.getTypeHandler().validate(col, this, errs);
    }
  }


  /**
    Inserts a new record into the database using data from
    the form.  Fields with no value in the form, and fields
    for which the renderForSQL method fails by returning null,
    are omitted from the field insertion list.  Corresponding
    Form fields should have the simple name of the column.
    <P>
    If the table has a primary key, this sets the corresponding form
    field to the database's "last insert ID" based on the possibility
    that it might be an autoincrement column.
    <P>
    Contains policy that any column named "registration" or
    "last_update" will be set to today's date on insert.
    TODO: This policy should be handled by extended schema info
    rather than column name.
    */
  public void insert(String tableName, Connection conn) {
    TableInfo table = schema.getTable(tableName);
    // checkForPrimaryKeyField();
    StringBuffer sqlbuf = new StringBuffer("insert into ");
    StringBuffer values = new StringBuffer();
    sqlbuf.append(table.getName());
    sqlbuf.append(" (");
    Enumeration cols = table.columns();
    boolean first = true;
    while (cols.hasMoreElements()) {
      ColumnInfo col = (ColumnInfo)cols.nextElement();
      String field = col.getName();
      // TODO: Decide what to do about this whole feature!
      if ("registration".equals(field) || "last_update".equals(field)) {
	String now = new java.sql.Date(System.currentTimeMillis()).toString();
	setValue(field, now);
      }
      String literal
	= col.getTypeHandler().renderForSQL(col, this, field);
      if (literal!=null) {
	if (first) {
	  first=false;
	} else {
	  sqlbuf.append(',');
	  values.append(',');
	}
	sqlbuf.append(field);
	values.append(literal);
      }
    }
    sqlbuf.append(")");
    sqlbuf.append(" values(");
    sqlbuf.append(values.toString());
    sqlbuf.append(")");

    String sql = null;
    try {
      Statement stmt = conn.createStatement();
      sql = sqlbuf.toString();
      dbgPrint("Executing: "+sql);
      stmt.executeUpdate(sql);
      String primary = table.getPrimaryKey();
      if (primary!=null) {
	setValue(primary, ""+Query.getLastInsertID(stmt));
      }
      stmt.close();
      // If there is an "afterInsertAction" for this table
      // (a Runnable), run it here.
    } catch(Exception ex) {
      throw new WrappedException(ex);
    }
  }


  /**
    Updates a record from a form.  Automatically modifies
    any fields named "last_update".  TODO: (This policy
    should really be established by schema information instead.)
    Columns where renderForSQL returns null do not appear
    in the generated update statement.
    <P>
    Enhancements for updateRecord and insertRecord:  These
    should be extended with table-specific hooks to perform
    additional actions such as updating lists of opportunity
    categories.  They
    could be enhanced to directly support updating of
    relationship tables, e.g. opportunity categories.
    <P>
    Returns true unless no such record exists.
    */
  public boolean update(String tableName, Connection conn) {
    TableInfo table = schema.getTable(tableName);
    // table.checkForPrimaryKeyField();
    String primary = table.getPrimaryKey();
    Object primaryValue = null;

    StringBuffer buf=new StringBuffer("update ");
    buf.append(table.getName());
    Enumeration cols = table.columns();
    boolean first = true;
    while (cols.hasMoreElements()) {
      ColumnInfo col = (ColumnInfo)cols.nextElement();
      String colname = col.getName();
      if ("last_update".equals(colname)) {
	long now = System.currentTimeMillis();
	setValue(colname, new java.sql.Date(now).toString());
      }
      String value
	= col.getTypeHandler().renderForSQL(col, this, colname);
      if (colname.equals(primary)) primaryValue = value;
      if (value!=null) {
        if (first) {
          buf.append(" set ");
          first = false;
        } else buf.append(',');
        buf.append(colname);
        buf.append('=');
        buf.append(value);
      }
    }
    buf.append(" where ");
    buf.append(primary);
    buf.append("=");
    buf.append(primaryValue);
    String sql = buf.toString();
    dbgPrint("Executing: "+sql);
    try {
      Statement stmt = conn.createStatement();
      int nrows = stmt.executeUpdate(sql);
      stmt.close();
      // If there is a "afterUpdateAction" (a Runnable) for this table,
      // run it here.
      return nrows!=0;
    } catch(SQLException ex) {
      System.err.println("Internal error: "+ex);
      // not reached:
      return true;
    }
  }


  /**
    Deletes the record determined by this form, using the
    given database connection.  Uses the given table's primary key
    to determine which record to delete.
    */
  public void delete(String tableName, Connection conn) {
    TableInfo table = schema.getTable(tableName);
    // table.checkForPrimaryKeyField();
    String sql =
      "delete from "+table.getName()
      +" as "+table.getShorthand()+" where "+primaryKeyCheck(table);
    dbgPrint("Executing: "+sql);
    // check1Row(conn.createStatement(), sql);
  }


  //// PRIVATE METHODS ////

  /**
    Builds and returns an array of the ColumnInfo objects for
    the fields of this table, plus ColumnInfos for the given
    "extra" fields.
    */
  private ColumnInfo[] allColumns(TableInfo table, String[] extraFields) {
    Enumeration tcols = table.columns();
    Vector columns = new Vector();
    while (tcols.hasMoreElements()) {
      columns.addElement(tcols.nextElement());
    }
    for (int j=0; j<extraFields.length; j++) {
      columns.addElement(schema.getColumn(extraFields[j]));
    }
    ColumnInfo[] cols = new ColumnInfo[columns.size()];
    columns.copyInto(cols);
    return cols;
  }


  /**
    Builds and returns an array of names of all the fields
    of this table, plus the names of the given "extra" fields.
    Only the "extra" field names are qualified.
    */
  private String[] allFieldNames(TableInfo table, String[] extraFields) {
    Enumeration tcols = table.columns();
    Vector columns = new Vector();
    while (tcols.hasMoreElements()) {
      columns.addElement(((ColumnInfo)tcols.nextElement()).getName());
    }
    for (int j=0; j<extraFields.length; j++) {
      columns.addElement(extraFields[j]);
    }
    String[] cols = new String[columns.size()];
    columns.copyInto(cols);
    return cols;
  }


  /**
     Concatenate two arrays of strings into a new one.
  */
  private String[] plusFieldNames(String[] names, String[] extras) {
    Vector v = new Vector();
    for (int i=0; i<names.length; i++) {
      v.addElement(names[i]);
    }
    for (int i=0; i<extras.length; i++) {
      v.addElement(extras[i]);
    }
    String[] all = new String[v.size()];
    v.copyInto(all);
    return all;
  }


  /**
    Generate and return SQL to compare the value in the form
    associated with the name of the primary key field
    against the database primary key field.
    */
  private String primaryKeyCheck(TableInfo table) {
    if (table.getExtends()!=null) {
      TableInfo parent = table.getExtends();
      ColumnInfo col = parent.getPrimaryColumn();
      String parentkey = col.getName();
      // We expect the form to already contain a value for the
      // parent table's primary key, in the right attribute.
      return table.getShorthand()+"."
	+table.getPrimaryKey()+"="+Query.sqlEscape(getValue(parentkey));
    } else {
      return table.getShorthand()
	+"."+table.getPrimaryKey()
	+"="+Query.sqlEscape(getValue(table.getPrimaryKey()));
    }
  }


  private static void dbgPrint(String msg) {
    if (debug) {
      System.err.println(msg);
    }
  }


  //// PRIVATE DATA ////

  // Database schema to bind Form data to.
  private final Schema schema;

  private static final String[] noStrings = {};


}
