// Copyright 1999-2001 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License.
// See http://www.gjt.org/doc/lgpl/license.html for details.

package com.perdues.db;

import java.util.*;
import java.sql.*;


/**
  Class that holds some clauses of a SQL select
  and helps you build select statements.  Serves a couple
  of different purposes.
  <P>
  First, this helps you build the syntax of the statement by
  letting you add information to any part of the statement at
  any time, and by inserting punctuation and logical connectives
  between the parts.
  <P>
  Second, this class builds SQL with properly escaped and quoted
  literals, so characters such as "'" can appear in your data.
  <P>
  Finally, if you add columns using methods that take ColumnInfo
  objects, this will automatically add the tables of those columns
  to the tables list.  If you request equijoins for a table, it will
  use foreign key information in the Schema to generate the conditions
  that the primary keys of tables refer to match the foreign key
  into it from the given table.
  */
public class Query {

  //// STATIC UTILITY METHODS ////

  /**
    Returns the last "insert ID" automatically generated by the
    database for an insert request on the given Statement's connection.
    The implementation is specific to the database software.
    */
  public static int getLastInsertID(Statement stmt) throws SQLException {
    ResultSet rs = stmt.executeQuery("select last_insert_id()");
    try {
      if (!rs.next())
	throw new SQLException("DBUtil.getLastInsertID: Empty ResultSet");
      int value = rs.getInt(1);
      System.err.println("Last insert was "+value);
      return value;
    } finally {
      rs.close();
    }
  }

  /**
    Escape the String so it is appropriate as part of a SQL string
    literal.
    */
  public static String sqlEscape(String s) {
    if (s.indexOf('\'')<0) return s;

    StringBuffer buf = new StringBuffer();
    for (int i=0; i<s.length(); i++) {
      char c = s.charAt(i);
      if (c=='\'') buf.append('\'');
      buf.append(c);
    }
    return buf.toString();
  }


  /**
    Formats an object as a literal for SQL.
    Currently handles Strings, Dates,
    java.sql.Dates, java.util.Dates, and Numbers.
    String formatting is specific to mySQL!!!
    */
  public static String sqlFormat(Object o) {
    if (o instanceof String) {
      String s = (String)o;
      StringTokenizer tok = new StringTokenizer(s, "\"'\\", true);
      StringBuffer buf = new StringBuffer("'");
      while (tok.hasMoreTokens()) {
	String t = tok.nextToken();
	if (t.length()==1) {
	  switch (t.charAt(0)) {
	  case '"':
	  case '\'':
	  case '\\':
	    buf.append('\\');
	    break;
	  }
	}
	buf.append(t);
      }
      buf.append("'");
      return buf.toString();
    } else if (o instanceof Number) {
      return o.toString();
    } else if (o instanceof java.sql.Date) {
      return "'"+o.toString()+"'";
    } else if (o instanceof java.util.Date) {
      return sqlFormat(new java.sql.Date(((java.util.Date)o).getTime()));
    } else throw new IllegalArgumentException
	     ("sqlFormat does not accept type "+o.getClass().getName());
  }


  //// PUBLIC DATA ////

  public static boolean debug = false;
  

  //// CONSTRUCTORS ////

  public Query(Schema schema) {
    this.schema = schema;
  }


  /**
    You can build a select statement from another Query.
    */
  public Query(Query template) {
    columns = new StringBuffer(template.columns.toString());
    tablelist = new StringBuffer(template.tablelist.toString());
    where = new StringBuffer(template.where.toString());
    having = new StringBuffer(template.having.toString());
    tables = (Vector)template.tables.clone();
    orderBy = template.orderBy;
    groupBy = template.groupBy;
    limit = template.limit;
  }


  //// PUBLIC METHODS ////

  /**
    Add a result column that is not a table column.  Simply
    adds this string to the comma-separated result columns.
    This may be any SQL expression suitable for the column list
    part of a Select.
    */
  public void addWhat(String what) {
    if (columns.length()>0)
      columns.append(',');
    columns.append(what);
  }


  /**
    Add a column to the list of columns to select, given
    its name; and adds the column's table to the set of
    tables for the query.
    <P>
    Same as addColumn(ColumnInfo col), but takes a String
    naming the column.  (A string naming the column
    is a qualified name with the table name or shorthand,
    and the name of the column, separated by ".".)
    */
  public void addColumn(String colName) {
    ColumnInfo col = schema.getColumn(colName);
    addColumn(col);
  }


  /**
    Add a column to the list of columns to select, and
    its table to the set of tables for the query.
    Use of the ColumnInfo object reduces potential ambiguity
    and automatically adds the column's table to the list
    of tables for the query.
    */
  public void addColumn(ColumnInfo col) {
    addTable(col.getTable());
    if (columns.length()>0)
      columns.append(',');
    columns.append(col.getFullName());
  }
  

  /**
    Adds each column ID String in the given array
    to this Query, as by addColumn.
    */
  public void addColumns(String[] colIds) {
    for (int i=0; i<colIds.length; i++) {
      addColumn(colIds[i]);
    }
  }


  /**
    Add a table by name to the list
    of tables to select from.  The name may be either
    a full name or a shorthand.
  */
  public void addTable(String tableName) {
    addTable(schema.getTable(tableName));
  }


  /**
    Add a table to the statement, using a TableInfo object.
    If the table has a shorthand name, this will establish
    the shorthand for use in the rest of the query.
    You may add the same table repeatedly through
    this method with no harmful effects.
    */
  public void addTable(TableInfo table) {
    if (debug) System.out.println("Adding table "+table);
    if (!tables.contains(table)) {
      tables.addElement(table);
      String t = table.getName();
      if (table.hasShorthand())
	t = t+" as "+table.getShorthand();
      if (tablelist.length()>0)
	tablelist.append(',');
      tablelist.append(t);
    }
  }


  /**
    Add a table to the statement, giving an alias.
    Does not add the table to the list of tables,
    so it gets no support when building joins.
    (Any other limitations from this?)
    */
  public void addAliasedTable(String name, String alias) {
    if (debug) System.out.println("Adding table "+name+" as "+alias);
    if (tablelist.length()>0)
      tablelist.append(',');
    tablelist.append(name);
    tablelist.append(" as ");
    tablelist.append(alias);
  }


  /**
    Add a condition to the "where" part of the select.
    The conditions are joined by "and".
    */
  public void addCondition(String expr) {
    if (where.length()>0) {
      where.append(" and ");
    } else {
      where.append(" where ");
    }
    where.append(expr);
  }


  /**
    Add a condition to the "having" part of the select.
    The conditions are joined by "and".
    */
  public void addHaving(String expr) {
    if (having.length()>0) {
      having.append(" and ");
    } else {
      having.append(" having ");
    }
    having.append(expr);
  }


  /**
    Adds a condition that the value of the named
    field must be equal to the given value, or
    if the field is an empty string or null no condition
    is added.  This is useful for building search
    queries from text fields in forms.
    */
  public void addSearchEqualCondition(String field, String value) {
    if (value==null || value.equals("")) return;
    else addCondition(field+"="+sqlFormat(value));
  }


  /**
    Adds a condition that the value of the named
    field must contain the given value, or
    if the field is an empty string or null no condition
    is added.  This is useful for building search
    queries from text fields in forms.
    */
  public void addSearchContainsCondition(String field, String value) {
    if (value==null || value.equals("")) return;
    else addCondition(field+" like "+sqlFormat("%"+value+"%"));
  }


  /**
    Adds a condition comparing the value of a field to
    a specific String value.
    */
  public void addFieldTest(String field, String op, String value) {
    addCondition(field+op+sqlFormat(value));
  }


  /**
    Add a table to the join across known tables.  Calls
    addEquiJoins with a TableInfo parameter.
    */
  public void addEquiJoins(String tableName) {
    addEquiJoins(schema.getTable(tableName));
  }

  /**
    Adds equijoin conditions to this Query, requiring that
    foreign keys of the given TableInfo must be equal to
    the corresponding primary key of the table referred to.
    <P>
    Better to do this just once per table before rendering
    the query.  Probably buggy if the given table has more than one column
    that is a foreign key into the the same table.
    */
  public void addEquiJoins(TableInfo table) {
    // Current implementation is kind of crazy.  Given table
    // shouldn't have to have a primary key, and better to
    // search all columns of the given table, checking the
    // type of each.
    debug("Joining from "+table);
    // ColumnInfo primary = table.getPrimaryColumn();
    // if (primary==null) err("No primary key for: "+table);
    for (int i=0; i<tables.size(); i++) {
      TableInfo t2 = (TableInfo)tables.elementAt(i);
      if (t2==table) continue;
      debug(" ... to "+t2);
      ColumnInfo p2 = t2.getPrimaryColumn();
      if (p2==null) continue;
      ColumnInfo fkey = table.getForeignKey(t2);
      if (fkey==null) continue;
      addCondition(fkey.getFullName()+"="+p2.getFullName());
    }
  }


  /**
    Sets contents for the "group by" clause of the statement.
    */
  public void setGroupBy(String clause) {
    groupBy = " group by "+clause;
  }


  /**
    Sets contents for the "order by" clause of the statement.
    */
  public void setOrderBy(String clause) {
    orderBy = " order by "+clause;
  }


  /**
    Sets contents for the (MySQL-specific) "limit" clause.
    */
  public void setLimit(String clause) {
    limit = " limit "+clause;
  }


  //// PUBLIC RENDERING METHODS ////

  /**
    Returns a string containing the complete
    select statement.
    */
  public String toString() {
    return render();
  }


  /**
    Returns a string containing the complete
    select statement.
    */
  public String render() {
    StringBuffer stmt = new StringBuffer("select ");
    stmt.append(columns.toString());
    stmt.append(" from ");
    stmt.append(tablelist.toString());
    stmt.append(where.toString());
    stmt.append(groupBy);
    stmt.append(having.toString());
    stmt.append(orderBy);
    stmt.append(limit);
    return stmt.toString();
  }


  //// Private methods ////


  private static void err(String s) {
    throw new RuntimeException(s);
  }


  private static void debug(String s) {
    if (debug) System.out.println(s);
  }


  //// PRIVATE DATA ////

  private Schema schema;

  // Contains TableInfo objects.
  private Vector tables = new Vector();

  private StringBuffer columns = new StringBuffer();

  private StringBuffer tablelist = new StringBuffer();

  private StringBuffer where = new StringBuffer();

  private StringBuffer having = new StringBuffer();

  // If non-null, contains the guts of the "group by" clause.
  private String groupBy = "";

  // If non-null, contains the guts of the "order by" clause.
  private String orderBy = "";

  private String limit = "";

}
