// Copyright 1999, 2000 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License.
// See http://www.gjt.org/doc/lgpl/license.html for details.

package com.perdues.db;

import java.util.*;
import java.io.*;
import java.lang.reflect.Array;
import com.perdues.PString;
import com.perdues.XP;
import com.perdues.XMLTag;
import com.perdues.db.typeHandlers.*;


/**
  This class defines information about one or more databases.  A
  Schema contains information about one or more databases, their
  tables and fields.  Beyond names and types, a Schema object contains
  information about which fields represent foreign keys and
  application-level type information about the fields.  For example a
  field type can be a high-level type such as a phone number or an
  email, not just a varchar or integer.
  <P>
  Schemas work with the Form class to provide very easy and concise
  database operations: read, insert, update, delete, and very automated
  validation.
  <P>
  These Schema objects are in principle immutable except that the
  current database can be changed with the "use" or "useDB" method.
  This immutability is not enforced, however.
  <P>
  The type information for a column can indicate a Java type or a
  conceptual type such as "email", stored as a standard database type
  such as VARCHAR, but validated in a more specific way, and may have
  one or more specialized presentations as HTML forms.
  <P>
  Validation for application types and certain other operations are
  determined by DBTypeHandler classes.  A Schema has a set of
  DBTypeHandlers, defined by addTypeHandler.

  @see DBTypeHandler
  */
public class Schema {


  //// CONSTRUCTORS ////

  /**
     The primary way to build a Schema is with Schema.configure,
     but you can build one similar to one that already exists with this
     constructor.
  */
  public Schema(Schema template) {
    for (Enumeration en = template.databases.keys(); en.hasMoreElements(); ) {
      String key = (String)en.nextElement();
      DBInfo db = (DBInfo)template.databases.get(key);
      databases.put(key, new DBInfo(this, db));
    }
    currentDatabase = (DBInfo)template.currentDatabase;
    handlers = (Hashtable)template.handlers.clone();
  }
  
  private Schema() {}


  //// PUBLIC STATIC METHODS ////

  /**
     Loads database definitions from a file or archive member, given
     its Java "resource name".  This sets the primary schema according
     to the information in the resource file.  If a description
     already exists in the primary Schema, this overwrites it.  The first
     database described in the resource file becomes the Schema's
     current database.
     <P>
     Java resource names are like relative file paths, with "/" always
     the delimiter between path elements.  The paths must be relative
     to some element of the CLASSPATH, and the first matching file is
     chosen.  For example, if CLASSPATH is
     /home/myself/classes:/foo/classes, and the the resourceName is
     "db/config.xml", then the search will first look for a file named
     /home/myself/classes/db/config.xml, then
     /foo/classes/db/config.xml.  If a CLASSPATH element is an archive
     file, the relative path indicates an archive member.
  */
  public static Schema configure(String resourceName) throws IOException {
    XMLTag[] tags = XP.loadResource(resourceName);
    Schema schema = new Schema();
    for (int i=0; i<tags.length; i++) {
      XMLTag tag = tags[i];
      schema.addDatabase(tag);
    }
    return schema;
  }


  //// PUBLIC METHODS ////

  /**
    Get a table from the current database by name
    or shorthand, or errs if there is none.
    */
  public TableInfo getTable(String name) {
    return currentDatabase.getTable(name);
  }


  /**
    Return a ColumnInfo given a qualified column name,
    from the default Database.
    */
  public ColumnInfo getColumn(String name) {
    return currentDatabase.getColumn(name);
  }


  /**
    Return the default Database.
    */
  public DBInfo getDB() {
    return currentDatabase;
  }


  /**
     Sets the current database.
  */
  public void useDB(String databaseName) {
    currentDatabase = get(databaseName);
  }


  /**
     Enumerate all available databases.
  */
  public Enumeration databases() {
    return databases.elements();
  }


  /**
    Return a Database given its name.  Throws an IllegalArgumentException
    if there is none.
    */
  public DBInfo get(String databaseName) {
    DBInfo s = (DBInfo)databases.get(databaseName);
    if (s==null)
      throw new IllegalArgumentException("No database named "+databaseName);
    return s;
  }


  /**
     Adds a database, defined by an XMLTag of type "db", to the set of
     databases described by this Schema.  Replaces any existing
     database with the same name.  If it is the first database added
     to this schema, it is set as the current database.
   */
  public void addDatabase(XMLTag tag) {
    if (!tag.getType().equals("db")) {
      throw new IllegalArgumentException
	("Can't define a database from a "+tag.getType()+" tag.");
    }
    String name = tag.requiredProperty("name");
    DBInfo database = new DBInfo(this, tag);
    databases.put(name, database);
    if (currentDatabase==null)
      currentDatabase = database;
  }


  /**
    Adds a handler for a type.
    */
  public void addTypeHandler(String type, DBTypeHandler handler) {
    handlers.put(type, handler);
  }


  /**
     Defines a global ("default") handler for a type.  Overrides an existing
     default.  Type handlers for a schema override it in turn.
  */
  public static void defTypeHandler(String type, DBTypeHandler handler) {
    defaultHandlers.put(type, handler);
  }


  //// PACKAGE-LEVEL METHODS ////

  DBTypeHandler getDBHandler(String typename) {
    // TODO: Define user-extensibility.
    //   Must support handler classes in user packages.
    //   In particular, needed to support JAR release of
    //   of this subsystem.  Probably list of 
    DBTypeHandler handler = (DBTypeHandler)handlers.get(typename);
    if (handler==null)
      throw new IllegalArgumentException
	("No DBHandler for type "+typename);
    return handler;
  }


  //// PRIVATE STATIC DATA ////


  // A default set of type handlers.  This maps from
  // type name string to handler object.
  private static Hashtable defaultHandlers = new Hashtable();

  // Table info is simply "table", "<name>"
  // Column info will change in the future, but today
  // is "column", "<name>", "<type>", "<ignored>", ...
  // Type Integer is a number, ID is an integer primary key.
  static {
    try {
      defTypeHandler("Date", new DateTypeHandler());
      defTypeHandler("Email", new EmailTypeHandler());
      defTypeHandler("ID", new IDTypeHandler());
      defTypeHandler("Integer", new IntegerTypeHandler());
      defTypeHandler("Numeric", new NumericTypeHandler());
      defTypeHandler("Phone", new PhoneTypeHandler());
      defTypeHandler("PhoneAC", new PhoneACTypeHandler());
      defTypeHandler("SSN", new SSNTypeHandler());
      defTypeHandler("String", new StringTypeHandler());
      defTypeHandler("Zipcode", new ZipcodeTypeHandler());
      // Faked handlers, not really implemented:
      defTypeHandler("YN", new StringTypeHandler());
      defTypeHandler("Time", new StringTypeHandler());
      defTypeHandler("Enum", new StringTypeHandler());
      defTypeHandler("URL", new StringTypeHandler());
      defTypeHandler("Password", new StringTypeHandler());
    } catch(Throwable ex) {
      ex.printStackTrace();
    }
  }


  //// PRIVATE DATA ////

  // Maps from database name to the Database object
  // for that database.
  private Hashtable databases = new Hashtable();

  // The currently-active database
  private DBInfo currentDatabase;

  // Maps from extended schema type names to a DBTypeHandler for each.
  private Hashtable handlers = (Hashtable)defaultHandlers.clone();

}
