// Copyright 1999, 2000 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License.
// See http://www.gjt.org/doc/lgpl/license.html for details.

package com.perdues.db;

import java.util.*;
import java.sql.*;
import java.util.Date;
import java.text.SimpleDateFormat;

import com.perdues.WrappedException;
import com.perdues.XMLTag;


/**
   Class holding information about a table of a database.
  */
public class TableInfo {

  //// PUBLIC DATA ////

  public static boolean debug = true;


  //// CONSTRUCTORS ////

  /**
     Constructs a TableInfo from a DBInfo and an XMLTag.
     The DBInfo becomes the database of the table, and the
     XMLTag, which must be of type "table", defines
     the attributes of the table.
     <P>
     XML attributes are:
     <UL>
     <LI>name (required)
     <LI>primaryKey (optional)
     <LI>extends (optional)
     <LI>shorthand (optional)
     </UL>
     Elements should be of type "field".
   */
  TableInfo(DBInfo db, XMLTag info) {
    if (!"table".equals(info.getType())) {
      throw new IllegalArgumentException
	("Within database tag, expecting 'table' tag instead of "
	 +info.getType());
    }
    database = db;
    name = info.requiredProperty("name");
    primaryKey = info.getProperty("primaryKey");
    extendTable = info.getProperty("extends");
    shorthand = info.getProperty("shorthand");
    if (shorthand==null) shorthand = name;
    XMLTag[] ctags = info.getElements();
    for (int i=0; i<ctags.length; i++) {
      XMLTag ctag = ctags[i];
      columns.put(ctag.requiredProperty("name"), new ColumnInfo(this, ctag));
    }
  }


  //// PUBLIC METHODS ////

  /**
    Unqualified name of the table.
    */
  public String getName() { return name; }


  /**
     Fully-qualified name of the table.
  */
  public String getFullName() {
    // Specific to MySQL.
    return database.getName()+"."+getName();
  }


  /**
     Adequately-qualified name of table based.  Same as fully
     qualified name unless there is more than one database and
     this table's database is not the one designated as in-use.
   */
  public String getEnoughName() {
    throw new IllegalStateException("Unimplemented");
  }
    


  /**
    Returns true iff this table has a shorthand.
    In this case getShorthand will return a different
    result than getName.
    */
  public boolean hasShorthand() {
    return shorthand!=name;
  }


  /**
    Shorthand name of the table.  If no shorthand name
    is supplied, this will give the same value as getName().
    */
  public String getShorthand() { return shorthand; }


  /**
    The Database of this table.
    */
  public DBInfo getDatabase() {
    return database;
  }


  /**
    Returns the ColumnInfo having the given, unqualified name.
    */
  public ColumnInfo getColumn(String name) {
    ColumnInfo col = (ColumnInfo)columns.get(name);
    if (col==null) System.err.println("No column named "+name);
    return col;
  }

  void setColumn(String name, ColumnInfo info) {
    columns.put(name, info);
  }


  /**
    Returns an Enumeration of the columns of this table
    as ColumnInfo objects.
    */
  Enumeration columns() {
    return columns.elements();
  }

  /**
    Returns the name of the primary key field, or null
    if there is none.
    */
  public String getPrimaryKey() { return primaryKey; }

  void setPrimaryKey(String id) {
    primaryKey = id;
  }


  /**
    Returns the column that is the primary key.
    */
  public ColumnInfo getPrimaryColumn() {
    if (primaryKey==null) return null;
    return (ColumnInfo)columns.get(primaryKey);
  }


  /**
    Returns the TableInfo of the table that this extends,
    or null if none.  Bad name throws an exception.
    */
  public TableInfo getExtends() {
    if (extendTable==null) return null;
    else return database.getTable(extendTable);
  }


  /**
    Returns the column that is the foreign key
    into the given Table, or null if none.
    */
  public ColumnInfo getForeignKey(TableInfo table) {
    String name = table.getName();
    Enumeration cols = columns.elements();
    while (cols.hasMoreElements()) {
      ColumnInfo col = (ColumnInfo)cols.nextElement();
      if (col.getType().equals(name))
        return col;
    }
    return null;
  }

  //// PRIVATE DATA ////

  // The database this is part of.
  DBInfo database;

  // The unqualified table name.
  String name;

  // The table's shorthand name.
  String shorthand;

  // Keys are Strings, values are ColumnInfos.
  private Hashtable columns = new Hashtable();

  // In no case does this statement have
  // the value of the primary key in it.
  // A Select statement object for reading from the database.
  // If a request indicates additional rows, we copy this and continue
  // defining the statement to be used.  
  private Query reader = null;

  // Name of the primary key field of this table, if any.
  private String primaryKey = null;

  // Name of the table this one "extends",
  // or null if none.
  private String extendTable;

}
