// Copyright 2000,2001 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with ABSOLUTELY NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License.
// See http://www.gjt.org/doc/lgpl/license.html for details.

package com.perdues.db;


/**
   Methods related to data validation.
  */
public class Validate {

  //// CONSTRUCTORS ////

  /**
     Class is not instantiable.
  */
  private Validate() {}


  //// STATIC METHODS ////

  /**
     A boolean in a form is valid if it is "0" or "1".
  */
  public static boolean okBoolean(String sBoolean) {
    return "0".equals(sBoolean) || "1".equals(sBoolean);
  }


  /**
     Rough validation of an email address.
     Checks for a domain ("@"), a "." in the domain
     part.  Only graphic ASCII characters pass this test.
     No angle brackets, parentheses, or double quotes;
     all are email address header syntax.
  */
  public static boolean okEmail(String email) {
    int atpos = email.lastIndexOf('@');
    if (atpos<0)
      return false;
    String domain = email.substring(atpos+1);
    int topdomainpos = domain.lastIndexOf('.');
    if (topdomainpos<0)
      return false;
    int l = email.length();
    for (int i=0; i<l; i++) {
      char ch = email.charAt(i);
      switch (ch) {
      case '"':
      case '<':
      case '>':
      case '(':
      case ')':
	return false;
      default:
	if (ch<=' ')
	  return false;
	if (128<=ch)
	  return false;
      }
    }
    return true;
  }

  
  /**
     Very generic phone number validator.  Allows digits,
     hyphens, parentheses, digits, and the letter 'x' to
     indicate an extension.  Any number of digits.
     Digits must be ASCII.
  */
  public static boolean okPhone(String phone) {
    return okPhoneDigits(phone)>0;
  }


  /**
     As okPhone, but returns the number of digits before
     the first "x", or a negative number if the input
     is not valid.
  */
  public static int okPhoneDigits(String phone) {
    int l = phone.length();
    // digits seen so far before the extension.
    int digits = 0;
    // Are we in the extension part:
    boolean extension = false;
    for (int i=0; i<l; i++) {
      char ch = phone.charAt(i);
      switch(ch) {
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
	if (!extension)
	  digits++;
	break;
      case 'x':
      case 'X':
	extension = true;
	break;
      case '(':
      case ')':
      case ' ':
	break;
      default:
	return -1;
      }
    }
    return digits;
  }

  /**
     Social Security Number (SSN) validator.  Requires exactly
     9 ASCII digits interspersed with spaces and dashes.
  */
  public static boolean okSSN(String ssn) {
    final int l = ssn.length();
    // Digits seen so far
    int digits = 0;
    for (int i=0; i<l; i++) {
      char ch = ssn.charAt(i);
      switch(ch) {
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
	  digits++;
	break;
      case ' ':
      case '-':
	break;
      default:
	return false;
      }
    }
    return digits==9;
  }

}
