// Copyright 1999-2001 Crispin Perdue <cris@perdues.com>
// 
// This is free software, and comes with ABSOLUTELY NO WARRANTY.
// You may distribute it under the terms of the Library GNU Public License
// as specified in the LICENSE.txt file that comes with this distribution.

package com.perdues.db.typeHandlers;

import java.util.*;
import com.perdues.db.*;


/*
  This type handler is designed for fields of numeric characters
  to be interpreted as character strings, not integers.  This is
  useful for data such as Zip codes, Social Security Numbers, phone
  numbers, etc., especially where leading zeros are appropriate.
  These are rendered to SQL as strings (surrounded by quotes).
 */
public class NumericTypeHandler extends StringTypeHandler implements DBTypeHandler {

  private int ndigits = -1;


  public NumericTypeHandler() {}

  /**
    You can build a Numeric type with a specific
    number of required digits.
    */
  public NumericTypeHandler(int digits) {
    ndigits = digits;
  }


  //// IMPLEMENTATION OF DBTypeHandler ////

  /**
    A numeric field must have exactly the specified number of digits,
    but if the number of digits is unspecified, then any nonzero number
    will do.
    */
  public void validate(ColumnInfo col, Form form, Hashtable errs) {
    String field = col.getName();
    String name = col.getFullName();
    String value = form.getValue(field);
    // Field is *absent*, ignore it.
    if (value==null)
      return;

    if (value.length()==0 && col.isRequired()) {
      errs.put(name, " is required.");
      return;
    }
    int lenspec = 1;

    // The number of digits is "defaulted" if the length is not
    // specified in the constructor nor in the XML tag.
    boolean defaulted = true;
    if (ndigits>0) {
      defaulted = false;
      lenspec = ndigits;
    } else {
      String tagdigits = col.get("digits");
      if (tagdigits!=null && tagdigits.length()>0) {
	lenspec = Integer.parseInt(tagdigits);
	defaulted = false;
      }
    }
    String msg = numericValidator(value, lenspec);
    if (msg==null && !defaulted && value.length()!=lenspec)
      // numericValidator said OK, but lengths don't match.
      msg = "Value must have "+lenspec+" digits.";
    if (msg!=null) {
      errs.put(name, msg);
    }
  }


  /**
    Returns a message string if the format is not valid, or
    null if OK.  The string must contain at least "min"
    characters, all numeric.
  */
  static public String numericValidator(String numeric, int min) {
    int len = numeric.length();
    for (int i=0; i<len; i++) {
      char ch = numeric.charAt(i);
      if (!Character.isDigit(ch)) {
	return "Value must be numeric.";
      }
    }
    if (len<min)
      return "Value must have "+min+" digits.";
    return null;
  }

}
