/* 
 * Copyright 1999-2001 Crispin Perdue.
 * Copyright (C) 1998  Ed Korthof and James Cooper
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * For more information about this software, visit:
 * http://www.bitmechanic.com/projects/
 */

//
// Summary of changes compared with com.bitmechanic.gsp.Form:
//
// * Supports any HttpServletRequest rather than a GspRequest.
// * Added support for radio buttons.
// * Method mergePathInfo supports .jsp as well as .gsp files.
// * Added default constructor for building Form objects from scratch.
// * Removed constructor that takes an initial nvHash.
// * Method getHidden generates an HTML hidden field for each value
//   of the form field.
// * Methods that take a FormLabel removed.  (Sorry if you liked this.)
// * Added keys() method and toQueryString() method.
// * Added toString method.
// * Method getDate can accept a single form field in standard
//   SQL date format (e.g. 2000-01-01).
// * Setting a field value to null removes it.  (Simplifies
//   setting a field to the same value as another field.)
// * Compared with older versions, new getValue method with a
//   default String to return if the form field is not present.
// * Added more Javadoc comments.
//

package com.perdues.web;

import java.util.*;
import java.text.*;
import javax.servlet.http.*;
import java.util.Date;


/**
 * This class is sort of modeled after Perl's CGI.pm.  It gives you a bunch
 * of methods that render HTML form elements with the state maintained.
 * In addition you can set the state of a field, which you can't do easily
 * with HttpServletRequest itself.
 * <p>
 * HTForm has a HttpServletRequest object, and consults that when rendering
 * a form element.  So you can use it in any case you would have used
 * HttpServletRequest.getParameterValues()
 * <p>
 * The database operations use extended database schema information from
 * the com.perdues.db.Schema class.
 *
 * @author James Cooper
 * @author Crispin Perdue <cris@perdues.com>
 */
public class HTForm {

  //// CONSTRUCTORS ////

  /**
    A form gets its initial state from the parameters in the request.
    */
  public HTForm(HttpServletRequest request) {
    this.request = request;
    for (Enumeration en = request.getParameterNames(); en.hasMoreElements(); ) {
      String key = (String)en.nextElement();
      nvHash.put(key, request.getParameterValues(key));
    }
  }


  /**
     Construct an empty Form.
  */
  public HTForm() {
    this.request = null;
  }


  /**
     Construct a copy of an existing HTForm.
  */
  public HTForm(HTForm form) {
    this.nvHash = (Hashtable)form.nvHash.clone();
    this.request = form.request;
  }


  ////  GET/SET METHODS FOR FORM NAME/VALUE PAIRS ////

  /**
   * A shorthand version of getValue(string, boolean) that passes in true
   * as the returnNull parameter.
   */
  public String getValue(String fieldname) {
    return getValue(fieldname, true);
  }

  /**
   * Returns the value of a given field.  If the field has more than one
   * value, it returns the first value on the list.  If the field has no
   * values, it returns the default, which may be null.
   */
  public String getValue(String fieldname, String defaultValue) {
    String list[] = getValues(fieldname);
    if(list == null || list.length < 1) {
      return defaultValue;
    } else
      return list[0];
  }


  /**
     Returns the value of the given field as an integer, throwing a
     NumberFormatException if the value does not represent an integer,
     as determined by Integer.parseInt.
  */
  public int getInt(String fieldname) {
    return Integer.parseInt(getValue(fieldname, ""));
  }


  /**
     Returns the value of the given field as an integer, returning
     the given defaultvalue if the value does not represent an intger
     as determined by Integer.parseInt.
  */
  public int getInt(String fieldname, int dfault) {
    try {
      return getInt(fieldname);
    } catch (NumberFormatException ex) {
      return dfault;
    }
  }


  /**
   * Returns the value of a given field.  If the field has more than one
   * value, it returns the first value on the list.  If the field has no
   * values and returnNull is true, it returns null.  Otherwise it returns
   * an empty string.
   */
  public String getValue(String fieldname, boolean returnNull) {
    return getValue(fieldname, returnNull ? null : "");
  }


  /**
     Returns true iff getValue(fieldname) would return a
     string with at least one character in it.
  */
  public boolean has(String fieldname) {
    return getValue(fieldname, "").length()>0;
  }


  /**
     True iff the given field of the form has a value
     equal to the one given.
   */
  public boolean hasValue(String fieldname, String value) {
    String list[] = getValues(fieldname);
    if(list != null) {
      for(int i = 0; i < list.length; i++) {
	if (list[i].equals(value))
	  return true;
      }
    }
    return false;
  }

  /**
   * Returns the list of values for the given fieldname.  If there is only
   * one value, this will be a single element array, just like in 
   * HttpServletRequest.getParameterValues().
   * <p>
   * The only difference between this method and the one in HttpServletRequest
   * is that this will consult a local hash of values first that can be
   * modified by adding, changing, or removing values.
   */
  public String[] getValues(String fieldname) {
    return (String[])nvHash.get(fieldname);
  }


  /**
   * Reconstructs a date from the three shadow fields.
   * If the shadow fields are not all there, reconstructs
   * it from a standard SQL date string in the given field.
   * Returns null if it cannot construct a Date from these.
   */
  public Date getDate(String fieldname) {
    Date date = null;
    try {
      String month = getValue(fieldname + "___month");
      String day = getValue(fieldname + "___day");
      String year = getValue(fieldname + "___year");

      if(month != null && day != null && year != null) {
	if(cal == null) cal = new GregorianCalendar();
	cal.set(Calendar.YEAR, Integer.parseInt(year));
	cal.set(Calendar.MONTH, Integer.parseInt(month));
	cal.set(Calendar.DAY_OF_MONTH, Integer.parseInt(day));
	date = cal.getTime();
      } else {
	// Will throw (an IllegalArgumentException) if bad format. -cp
	String d = getValue(fieldname);
	if (d!=null)
	  date = java.sql.Date.valueOf(d);
      }
    } catch(Exception e) { }
    
    return date;
  }

  /**
   * Splits "str" into an array of Strings using \r\n as the delimitter.
   * Useful for getting each line of a textarea back as a separate String
   */
  public String[] splitString(String str) {
    StringTokenizer st = new StringTokenizer(str, "\r\n");
    String vals[] = new String[st.countTokens()];
    for(int x = 0; st.hasMoreTokens(); x++) {
      vals[x] = st.nextToken();
    }
    return vals;
  }

  /**
   * Sets value in the given slot fieldname.  Note that this will *not* change
   * the value returned by HttpServletRequest, but it will be returned by
   * form.getValues() or form.getValue().  Setting to null removes any existing
   * values.  After setting to null, getValue will return null.
   */
  public void setValue(String fieldname, String value) {
    if(fieldname != null) {
      if (value==null) {
	removeValue(fieldname);
      } else {
	String list[] = new String[1];
	list[0] = value;
	nvHash.put(fieldname, list);
      }
    }
  }

  /**
   * Same as form.setValue() except it sets multiple values
   * from an array of values instead of taking a single value.
   * The new values replace any existing values.
   */
  public void setValues(String fieldname, String values[]) {
    if (fieldname != null && values != null) nvHash.put(fieldname, values);
  }


  /**
     Adds the given value to the list of values for the given
     field name if not already in the list.
  */
  public void addValue(String fieldname, String value) {
    String[] values = (String[])getValues(fieldname);
    if (values==null) {
      setValues(fieldname, new String[] {value});
    } else {
      for (int i=0; i<values.length; i++) {
	if (values[i].equals(value))
	  return;
      }
      String[] newvalues = new String[values.length+1];
      System.arraycopy(values, 0, newvalues, 0, values.length);
      newvalues[values.length] = value;
      setValues(fieldname, newvalues);
    }
  }


  /**
     Sets the given value if there is none yet for the given
     field name.
  */
  public void setDefault(String fieldname, String value) {
    if (getValue(fieldname)==null)
      setValue(fieldname, value);
  }


  /**
     Removes any occurrences of the given value from the list of
     values for fieldname.
  */
  public void removeValue(String fieldname, String value) {
    String[] values = (String[])getValues(fieldname);
    if (values==null)
      return;
    Vector v = new Vector();
    for (int i=0; i<values.length; i++) {
      String oldv = values[i];
      if (!oldv.equals(value))
	v.addElement(oldv);
    }
    String[] newvalues = new String[v.size()];
    v.copyInto(newvalues);
  }
	

  private static SimpleDateFormat fmt
    = new SimpleDateFormat("yyyy-MM-dd", Locale.US);

  /**
   * Sets a SQL-style string and three shadow variables for the date.
   * This lets special "Date widgets" pick up the right values, and
   * also allows straightforward programmatic access and hidden
   * fields that use this fieldname.
   */
  public void setDate(String fieldname, Date date) {
    if(cal == null) cal = new GregorianCalendar();
    cal.setTime(date);
    int month = cal.get(Calendar.MONTH);
    int day = cal.get(Calendar.DAY_OF_MONTH);
    int year = cal.get(Calendar.YEAR);
    String formatted;
    // SimpleDateFormat objects are not synchronized, believe it or not.
    synchronized (fmt) {
      formatted = fmt.format(date);
    }
    setValue(fieldname+"___month", Integer.toString(month));
    setValue(fieldname+"___day", Integer.toString(day));
    setValue(fieldname+"___year", Integer.toString(year));
    setValue(fieldname, formatted);
  }


  //// PUBLIC METHODS ////

  /**
     Create and return an HTTP query string (without the "?")
     representing the same field information contained in this HTForm.
  */
  public String toQueryString() {
    StringBuffer buf = new StringBuffer();
    boolean first = true;
    for (Enumeration en = nvHash.keys(); en.hasMoreElements(); ) {
      String key = (String)en.nextElement();
      String[] values = (String[])nvHash.get(key);
      for (int i=0; i<values.length; i++) {
	String v = values[i];
	if (first)
	  first = false;
	else
	  buf.append('&');
	buf.append(java.net.URLEncoder.encode(key));
	buf.append('=');
	buf.append(java.net.URLEncoder.encode(v));
      }
    }
    return buf.toString();
  }


  /**
   * Copies the contents of form into this form. If overwrite is true, then
   * any values set in both forms will be replaced with the contents of the
   * passed in form.  Otherwise the old value in the current form will be
   * retained.
   */
  public void mergeForm(HTForm form, boolean overwrite) {
    mergeEnum(form.nvHash.keys(), form, overwrite);
  }

  private void mergeEnum(Enumeration e, HTForm form, boolean overwrite) {
    while(e.hasMoreElements()) {
      String fieldname = (String)e.nextElement();
      if(overwrite || (getValue(fieldname) == null)) {
	String list[] = (String[])form.getValues(fieldname);
	setValues(fieldname, list);
      }
    }
  }

  /**
   * Treats PATH_INFO like another QUERY_STRING and folds those values into
   * the form.  This is useful if you want to pass dynamic information around,
   * but still want search engines to index your pages.  Most of them ignore
   * pages with ? in the URL
   */
  public void mergePathInfo(boolean overwrite) {
    String pathInfo = request.getRequestURI();
    if (pathInfo==null)
      return;

    // Deal with servlet engines that don't accurately
    // distinguish pathInfo from other parts of the path.
    String[] patterns = {".gsp/", ".jsp/"};
    for (int i=0; i<patterns.length; i++) {
      String pat = patterns[i];
      int pos = pathInfo.indexOf(pat);
      if (pos>=0) {
	pathInfo = pathInfo.substring(pos+pat.length()-1);
	break;
      }
    }

    // Real CGI-style pathInfo's typically begin with "/"
    // Our convention is to put the query info right after the first "/"
    if (pathInfo.startsWith("/"))
      pathInfo = pathInfo.substring(1);
    // Don't look beyond the second "/"
    int pos = pathInfo.indexOf('/');
    if(pos != -1) {
      pathInfo = pathInfo.substring(0, pos);
    }

    // Parse up the name/value pairs.
    if(pathInfo != null && pathInfo.length() > 0
       && pathInfo.indexOf('=') != -1) {
      Hashtable hash = HttpUtils.parseQueryString(pathInfo);
      for(Enumeration e = hash.keys(); e.hasMoreElements();) {
	String fieldname = (String)e.nextElement();
	if(overwrite || (getValue(fieldname) == null)) {
	  String list[] = (String[])hash.get(fieldname);
	  setValues(fieldname, list);
	}
      }
    }
  }

  /**
   * Removes the value with the specified key.  Returns the object that was
   * in that slot, or null if no value exists with that key.
   */
  public Object removeValue(String key) {
    return nvHash.remove(key);
  }


  /**
     Returns an enumeration of all the field names in this HTForm.
  */
  public Enumeration keys() {
    return nvHash.keys();
  }


  ///////////////////////////////////////////////////////////////////////////
  //
  // HTML WIDGET RENDERING
  //
  ///////////////////////////////////////////////////////////////////////////


  //// TEXTFIELD, PASSWORD, TEXTAREA ////

  /**
     Returns the same value, escaped for HTML, and suitably
     escaped to be inside an HTML attribute (quotes escaped).
     Use this to define text inside quotes in the value of
     an input field.
  */
  public String getText(String text) {
    return escapeValue(text);
  }


  /**
   * Returns a &lt;input type="text"&gt; HTML form element with the state
   * maintained
   */
  public String getTextfield(String fieldname, int size, int maxlength) {
    StringBuffer buffer = new StringBuffer(128);
    buffer.append("<input type=\"text\" size=\"");
    buffer.append(size);
    buffer.append("\" maxlength=\"");
    buffer.append(maxlength);
    buffer.append("\" value=\"");
    buffer.append(escapeValue(getValue(fieldname, false)));
    buffer.append("\" name=\"");
    buffer.append(fieldname);
    buffer.append("\">");
    return buffer.toString();
  }

  /**
   * Returns a &lt;input type="password"&gt; HTML form element with the state
   * maintained
   */
  public String getPassword(String fieldname, int size, int maxlength) {
    StringBuffer buffer = new StringBuffer(128);
    buffer.append("<input type=\"password\" size=\"");
    buffer.append(size);
    buffer.append("\" maxlength=\"");
    buffer.append(maxlength);
    buffer.append("\" value=\"");
    buffer.append(escapeValue(getValue(fieldname, false)));
    buffer.append("\" name=\"");
    buffer.append(fieldname);
    buffer.append("\">");
    return buffer.toString();
  }

  /**
   * Returns a &lt;textarea&gt; HTML form element with the state
   * maintained.
   *
   * @param fieldname name of textarea to create
   * @param rows number of rows in textarea
   * @param cols number of columns in textarea
   * @param wrap string to place in wrap="" attribute.  valid values
   *        are "none", "virtual", "physical"
   */
  public String getTextarea(String fieldname, int rows, int cols,
			    String wrap) {
    return "<textarea name=\"" + fieldname + "\" rows=\"" + rows + 
      "\" cols=\"" + cols + "\" wrap=\"" + wrap + "\">" +
      escapeValue(getValue(fieldname, false)) + "</textarea>";
  }

  /**
   * calls getTextarea() with "none" as the wrap attribute.
   */
  public String getTextarea(String fieldname, int rows, int cols) {
    return getTextarea(fieldname, rows, cols, "none");
  }


  //// CHOICE (PULL-DOWN MENU) AND LIST (SCROLLING) ////

  /**
     Render a choice (dropdown list) for the named field with
     the listed values.  This renders options as with getOptions.
   */
  public String getChoice(String fieldname, String[] values) {
    return getChoice(fieldname, values, emptyHash);
  }

  /**
     Render a choice (dropdown list) for the named field with
     the listed values.  This renders options as with getOptions.
   */
  public String getChoice(String fieldname, String values[], Hashtable labels) {
    return getList(fieldname, 0, false, values, labels);
  }

  /**
     Render a scrolling list for the named field and values,
     with the specified size in rows, and allowing multiple selection
     if allowMultiple is true.  This renders options as with getOptions.
  */
  public String getList(String fieldname, int size, boolean allowMultiple,
			String[] values) {
    return getList(fieldname, size, allowMultiple, values);
  }

  /**
     Render a scrolling list for the named field and values,
     with the specified size in rows, and allowing multiple selection
     if allowMultiple is true.  This renders options as with getOptions.
  */
  public String getList(String fieldname, int size, boolean allowMultiple,
			String values[], Hashtable labels) {
    StringBuffer buffer = new StringBuffer(512);
    buffer.append("<select name=\"" + fieldname + "\"");
    if(size > 0) buffer.append(" size=\"" + size + "\"");
    if(allowMultiple) buffer.append(" multiple");
    buffer.append(">\n");
    buffer.append(getOptions(fieldname, values, labels));
    buffer.append("</select>\n");
    
    return buffer.toString();
  }


  /**
     Return a list of option tags for an HTML select.  The values
     determine the values of the options.  Any value that maps
     to a value in the labels hash table will display as the value
     from the hash table.  Any value that appears in the HTML
     data of the form as a value of the field name is rendered as
     selected.  It will be selected more than once if it appears
     in the form data more than once.
   */
  public String getOptions(String fieldname, String values[], Hashtable labels) {
    StringBuffer buffer = new StringBuffer();
    for(int i = 0; i < values.length; i++) {
      buffer.append("<option value=\"" + escapeValue(values[i]));
      if(hasValue(fieldname, values[i])) buffer.append("\" selected>\n");
      else buffer.append("\">\n");
      if(labels.get(values[i]) != null) 
	buffer.append(labels.get(values[i]).toString());
      else buffer.append(values[i]);
    }
    return buffer.toString();
  }



  //// CHECKBOXES AND RADIO BUTTONS ////

  /**
    Renders a single checkbox with the given field name and
    value when checked.  Render your own label.  If this Form
    object's has the given value for the fieldname, render
    the checkbox as checked.
    */
  public String getCheckbox(String fieldname, String value) {
    return "<input type=checkbox name=\""+fieldname
      +"\" value=\""+escapeValue(value)+"\""
      +(value.equals(getValue(fieldname)) ? " checked" : "")+">";
  }

  /**
    Renders a single checkbox with the given field name and
    the value "y" when checked.  Render your own label.  If this Form
    object's has the given value for the fieldname, render
    the checkbox as checked.
  */
  public String getCheckbox(String fieldname) {
    return getCheckbox(fieldname, "y");
  }

  /**
    Renders a list of checkboxes, all with the same field name,
    with the values and labels the same, as if with an empty hash table
    in the variant with a hash table argument.
    */
  public String getCheckboxes(String fieldname, String[] values) {
    return getCheckboxes(fieldname, values, emptyHash);
  }

  /**
    Renders a list of checkboxes, all with the same field name, one
    for each value in the list.  Each checkbox is labeled with
    the HTML associated in the hash table with the value in the list.
    If no label is associated with the value, renders no HTML label
    onto the page.
    <P>
    Renders checkboxes checked based on the value or values for
    the given field in this Form object.  Renders each box with
    a checkmark if its values is equal to some value for
    the given field in this Form.
    */
  public String getCheckboxes(String field, String values[], Hashtable labels) {
    return getRadioOrCheckbox("checkbox", field, values, labels);
  }
    
  /**
    Renders a list of radio buttons, all with the same field name, one
    for each value in the list.  Each radio button is labeled with
    the HTML associated in the hash table with the value in the list.
    If no label is associated with the value, renders no HTML label
    onto the page.
    <P>
    Renders radio buttons checked based on the value for
    the given field in this Form object.  Renders a  box with
    a checkmark if its values is equal to the value for
    the given field in this Form.
  */
  public String getRadios(String fieldname, String values[], Hashtable labels) {
    return getRadioOrCheckbox("radio", fieldname, values, labels);
  }


  private String getRadioOrCheckbox(String type, String fieldname,
				    String[] valuesAndLabels) {
    final String[] vl = valuesAndLabels;
    if (vl.length%2!=0)
      throw new IllegalArgumentException
	("Radio or Checkbox: values/labels list length is uneven.");
    final int size = vl.length/2;
    String[] values = new String[size];
    Hashtable labels = new Hashtable(size);
    for (int i=0; i<size; i++) {
      values[i] = vl[2*i];
      labels.put(values[i], vl[2*i+1]);
    }
    return getRadioOrCheckbox(type, fieldname, values, labels);
  }

  private String getRadioOrCheckbox(String type, String fieldname, 
				    String values[], Hashtable labels) {
    StringBuffer buffer = new StringBuffer(512);

    for(int i = 0; i < values.length; i++) {
      buffer.append("<input type=\"");
      buffer.append(type);
      buffer.append("\" value=\"");
      buffer.append(escapeValue(values[i]));
      buffer.append("\" name=\"");
      buffer.append(fieldname);
      if(hasValue(fieldname, values[i])) buffer.append("\" checked>");
      else buffer.append("\">");
      if(labels.get(values[i]) != null) 
	buffer.append(labels.get(values[i]).toString());
      else buffer.append(values[i]);
    }

    return buffer.toString();        
  }


  //// HIDDEN FIELDS ////

  /**
   * Returns a set of &lt;input type="hidden"&gt; HTML form fields with the values
   * properly set. This is useful for maintaining a value or values posted
   * to the page from another form.  If the field has just one value, this
   * returns exactly one hidden field tag.
   */
  public String getHidden(String fieldname) {
    String[] values = getValues(fieldname);
    if (values==null)
      return "";
    StringBuffer buffer = new StringBuffer(128);
    for (int i=0; i<values.length; i++) {
      buffer.append("<INPUT type=\"hidden\" name=\"");
      buffer.append(fieldname);
      buffer.append("\" value=\"");
      buffer.append(escapeValue(values[i]));
      buffer.append("\">");
    }
    return buffer.toString();
  }


  //// DATE FIELD ////

  public String getDateField(String fieldname, int start_year, int stop_year) {
    // build the pull-down of years
    if(start_year > stop_year) start_year = stop_year;
    int num = stop_year - start_year + 1;
    String yearValues[] = new String[num];
    for(int i = 0; i < num; i++) {
      yearValues[i] = Integer.toString(start_year + i);
    }

    StringBuffer buffer = new StringBuffer(1024);
    buffer.append(getChoice(fieldname + "___month", monthValues, monthLabels));
    buffer.append(getChoice(fieldname + "___day", dayValues, dayLabels));
    buffer.append(getChoice(fieldname + "___year", yearValues));

    return buffer.toString();
  }


  //// OBJECT PROTOCOL ////

  /**
     Overridden method from class Object.
  */
  public String toString() {
    StringBuffer buf = new StringBuffer("<");
    buf.append(afterLast(getClass().getName(), "."));
    buf.append(' ');
    buf.append(toQueryString());
    buf.append(">");
    return buf.toString();
  }


  //// PRIVATE ////

  /**
   *  Escapes ", &, <, and > contained in the string with their
   *  ISO 8859-1 value.  Although < and > don't need to be escaped
   *  inside of quoted strings, we escape them to hide <!--#
   *  constructs from anything that might try to interpret them
   *  as server-side includes.  And they need to be escaped in the
   *  text of a textarea.  And some browsers might suck and get
   *  confused by < and > in quoted strings.
   */
  private String escapeValue(String str) {
    str = replace(str, '&', "&amp;");
    str = replace(str, '"', "&quot;");
    str = replace(str, '<', "&lt;");
    str = replace(str, '>', "&gt;");
    return str;
  }

  private static String replace(String str, char ch, String replace) {
    int pos = str.indexOf(ch);
    if(pos == -1) return str;
    StringBuffer buff = new StringBuffer(str.length() + 32);
    int start = 0;
    while(pos != -1 && start < str.length()) {
      buff.append(str.substring(start, pos));
      buff.append(replace);

      start = pos + 1;
      if(start < str.length()) pos = str.indexOf(ch, start);
    }
    if(start < str.length()) buff.append(str.substring(start));

    return buff.toString();
  }


  /**
     Returns a suffix of the subject, starting just after the
     last occurrence of the delimiter string.  If the delimiter
     is not found, returns the entire subject without copying it.
  */
  private static String afterLast(String subject, String delimiter) {
    int pos = subject.lastIndexOf(delimiter);
    if (pos<0) {
      return subject;
    } else {
      return subject.substring(pos+delimiter.length());
    }
  }


  //// PRIVATE DATA ////

  private HttpServletRequest request;
  private Hashtable nvHash = new Hashtable();
  private GregorianCalendar cal = null;
  private static final String emptyString = new String();

  private static final Hashtable emptyHash = new Hashtable(0);

  static  String monthNames[] = {
    "January", "February", "March", "April", "May", 
    "June", "July", "August", "September", "October",
    "November", "December" };
  static String dayValues[] = new String[31];
  static String monthValues[] = new String[12];
  static Hashtable monthLabels = new Hashtable();
  static Hashtable dayLabels = new Hashtable();

  static {
    for(int i = 1; i <= 31; i++) { 
      dayValues[i - 1] = Integer.toString(i);
    }
    for(int i = 0; i < 12; i++) {
      monthValues[i] = Integer.toString(i);
      monthLabels.put(monthValues[i], monthNames[i]);
    }
  }

}
